
/* 
  Copyright (c) 2002 Gavin E. Crooks <gec@compbio.berkeley.edu>

  Permission is hereby granted, free of charge, to any person obtaining a 
  copy of this software and associated documentation files (the "Software"),
  to deal in the Software without restriction, including without limitation
  the rights to use, copy, modify, merge, publish, distribute, sublicense,
  and/or sell copies of the Software, and to permit persons to whom the
  Software is furnished to do so, subject to the following conditions:

  The above copyright notice and this permission notice shall be included
  in all copies or substantial portions of the Software.

  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR 
  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, 
  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
  AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER 
  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING 
  FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER 
  DEALINGS IN  THE SOFTWARE.

  (This is the MIT Open Source License, 
  http://www.opensource.org/licenses/mit-license.html)
*/



/* ****************************************************************
  The multivariate negative hypergeometric distribution.

  This is the pdf of data drawn from a multinomial distribution
  with a Dirichlet prior.
**************************************************************** */


#include <math.h>
#include <assert.h>

#include <gsl/gsl_math.h>
#include <gsl/gsl_sf.h>


#include "nhypergeometric.h"

double
multivariate_negative_hypergeometric_pdf(const size_t K, 
					 const double x[], 
					 const double d[]) {
  return exp( multivariate_negative_hypergeometric_lnpdf(K, x, d) );
}


/* The log of the  multivariate negative hypergeometric pdf. 
   Use for large K, where the probablities will be very small.
*/
double
multivariate_negative_hypergeometric_lnpdf(const size_t K, 
					   const double x[], 
					   const double d[])
{
  size_t k;
  double log_p = 0.0;
  double X = 0.0;
  double D  =0.0;

  for(k = 0; k < K; k++) {
    assert( d[k] >= 0.0);
    assert( x[k] >= d[k]);
    X += x[k];
    D += d[k];
  }

  for(k = 0; k < K; k++) {
    //printf("x d : %g, %g\n", x[k], d[k]);
    if( x[k]==0.0 && d[k] ==0.0) continue;
    log_p += gsl_sf_lngamma(x[k]);
    log_p -= gsl_sf_lngamma(d[k]);
    log_p -= gsl_sf_lngamma(x[k] - d[k] + 1); 
  }

  log_p -= gsl_sf_lngamma(X + 1.);
  log_p += gsl_sf_lngamma(D + 1.);
  log_p += gsl_sf_lngamma(X - D + 1.);
  
  return log_p;
}



/* Convert Dirichlet and Multinomial parameters into the
   parameteres of the corresposing multivariant negative
   hypergeometric pdf 

   beta is the sum of the Dirichlet alpha parametes
   mean_theta[] is the average of each Dirichlet variable
   Thus mean_theta[k] = alpha[k]/ beta
*/
void 
dirichlet_to_nhyper(const size_t K, 
		    const double data[], 
		    const double mean_theta[], 
		    const double beta,  
		    double x[], 
		    double d[])
{
  int k;
  for(k=0; k<K; k++) {
    x[k] = data[k] + beta*mean_theta[k];
    d[k] = beta * mean_theta[k];
  }
}


