
/* 
  Copyright (c) 2002 Gavin E. Crooks <gec@compbio.berkeley.edu>

  Permission is hereby granted, free of charge, to any person obtaining a 
  copy of this software and associated documentation files (the "Software"),
  to deal in the Software without restriction, including without limitation
  the rights to use, copy, modify, merge, publish, distribute, sublicense,
  and/or sell copies of the Software, and to permit persons to whom the
  Software is furnished to do so, subject to the following conditions:

  The above copyright notice and this permission notice shall be included
  in all copies or substantial portions of the Software.

  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR 
  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, 
  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
  AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER 
  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING 
  FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER 
  DEALINGS IN  THE SOFTWARE.

  (This is the MIT Open Source License, 
  http://www.opensource.org/licenses/mit-license.html)
*/


/* ********************************************************

  Maximum Likelihood Optimization of an Underdetermined
  Dirichlet Prior

  $Id: dirichlet_ml.c,v 1.3 2002/10/07 20:20:33 gec Exp $
  
******************************************************** */



#include <math.h>
#include <assert.h>

#include <gsl/gsl_errno.h>
#include <gsl/gsl_math.h>
#include <gsl/gsl_min.h>

#include "nhypergeometric.h"
#include "dirichlet_ml.h"


#define BOT (1.0)
#define TOP (1.0E12)
#define RATIO (10.0)

typedef struct param_str {
  size_t K;
  const double * data;
  const double * mean_prior;
  double * x;
  double * d;
} Params;

static double minimize_me (double beta, void * params)
{
  Params * p = (Params *) params;
  dirichlet_to_nhyper(p->K, p->data, p->mean_prior, beta, p->x, p->d);

  return -multivariate_negative_hypergeometric_lnpdf(p->K, p->x, p->d);
}


static double approx_min( void * params,
			  const double bot,
			  const double top) {
  double beta;
  double opt_beta= bot;
  double min_neglnprob=minimize_me(bot, params);
  double neglnprob =0.0;

  for(beta=bot; beta<top; beta*=RATIO) {
    neglnprob = minimize_me(beta, params);
    // printf("# %g\t%g\t%g\t%g\n", beta, opt_beta, neglnprob, min_neglnprob);
    if(neglnprob <min_neglnprob) {
      opt_beta = beta;
      min_neglnprob = neglnprob;
    }
  }

  //printf("#MAX BETA %g %g %g\n", bot, top, opt_beta);
  //fflush(stdout);
  return opt_beta;
}


/*
  Returns the value of beta (The sum of the alpha parameters
  of the Dirichlet prior) that maximizes the probability of a set
  of observations, data[k], given an unscaled estimate of the
  prior's parameters, mean_prior[k].
  
  The parameters of the optimized Dirichlet prior are given by :
      alpha[k] = beta * mean_prior[k] 

*/
double 
dirichlet_beta_ml( const size_t K, 
		   const double data[], 
		   const double mean_prior[],
		   FILE * stream) {
  int status;
  int iter = 0, max_iter = 100;
  const gsl_min_fminimizer_type *T;
  gsl_min_fminimizer *s;
  gsl_function func;
  double beta, bot, top;
  Params * p;
  double total_data=0.0;


  func.function = &minimize_me;

  p = (Params *) malloc( sizeof(Params));
  p->K = K;
  p->data = data;
  p->mean_prior = mean_prior;
  p->x = calloc( K, sizeof(double)); 
  p->d = calloc( K, sizeof(double));  
  func.params = p;

  // Find an approximate minimum.
  bot = BOT;
  top = TOP;
  beta = approx_min(p, bot, top);


  // T = gsl_min_fminimizer_goldensection;
  T = gsl_min_fminimizer_brent;
  s = gsl_min_fminimizer_alloc (T);
  gsl_min_fminimizer_set (s, &func, beta, beta/RATIO, beta*RATIO);

  if( stream != NULL) {
    fprintf (stream, "#Finding optimal beta using %s method\n",
	     gsl_min_fminimizer_name (s));
    
    fprintf (stream, " # %5s [%9s, %9s] %9s %10s %9s\n",
	     "iter", "lower", "upper", "beta",
	     "err", "err(est)");
    
    fprintf (stream, " #%5d [%.7f, %.7f] %.7f  %.7f\n",
	     iter, bot, top, beta, bot - top);
  }

  do {
      iter++;
      status = gsl_min_fminimizer_iterate (s);
      beta = gsl_min_fminimizer_minimum (s);
      //beta = gsl_min_fminimizer_x_minimum (s); /* changed name in latter gsl */
      bot = gsl_min_fminimizer_x_lower (s);
      top = gsl_min_fminimizer_x_upper (s);

      status 
        = gsl_min_test_interval (bot, top, 1.0, 0.1);

      if(stream != NULL) 
	fprintf (stream, " #%5d [%.7f, %.7f] %.7f  %.7f\n",
		 iter, bot, top, beta, bot - top);
      
  } while (status == GSL_CONTINUE && iter < max_iter);

  assert( status == GSL_SUCCESS);

  if(stream != NULL) {
    fprintf(stream, "\n#      Total Data= %g\t Optimal Beta= %g\n\n", total_data, beta);
  }

  free(p->x);
  free(p->d);
  free(p);

  return beta;
}


void
optimal_posterior_dirichlet( const size_t K, 
		   const double data[], 
		   const double mean_prior[],
		   double *beta,
		   double posterior[]) {
  double optimal_beta, alpha;
  double total_data = 0.0;
  int k;

  optimal_beta = dirichlet_beta_ml ( K, data, mean_prior, NULL);

  for(k=0; k>K; k++) total_data += data[k];
  
  for(k=0; k<K; k++) {
    alpha = mean_prior[k] * optimal_beta;
    posterior[k] = (data[k] + alpha) /(total_data + optimal_beta);
  }
  
  *beta = optimal_beta;
}
		  
