/* 
  Copyright (c) 1997-2003 Gavin E. Crooks <gec@compbio.berkeley.edu> 
		     Univ. of California, Berkeley

  Permission is hereby granted, free of charge, to any person obtaining a 
  copy of this software and associated documentation files (the "Software"),
  to deal in the Software without restriction, including without limitation
  the rights to use, copy, modify, merge, publish, distribute, sublicense,
  and/or sell copies of the Software, and to permit persons to whom the
  Software is furnished to do so, subject to the following conditions:

  The above copyright notice and this permission notice shall be included
  in all copies or substantial portions of the Software.

  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR 
  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, 
  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
  AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER 
  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
  OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN 
  THE SOFTWARE.

  (This is the MIT Open Source License, 
  http://www.opensource.org/licenses/mit-license.html)

*/


/* *******************************************************************
   Computational Biology Toolkit - Utilities

   $Id: cbt_util.h,v 1.1 2003/08/20 22:03:42 gec Exp $
******************************************************************** */


#ifndef __CBT_UTIL_H__
#define __CBT_UTIL_H__

#include <stdlib.h>
#include <stdbool.h>

// Constants

//! Intended for internal use. Taken from GSL
#define __CBT_DBL_EPSILON__ (.2204460492503131e-16)

extern const double cbt_bits_per_nat;
extern const int cbt_line_length ;
extern const int cbt_line_length_max ;


//! Traditional MAX macro.
#ifndef MAX
#define MAX(x,y) (x>y ? x : y)
#endif

//! Traditional MIN macro.
#ifndef MIN
#define MIN(x,y) (x<y ? x : y)
#endif



//! Error handling
enum {
  CBT_CONTINUE_ON_ERROR, //< Quitly return error code
  CBT_WARN_ON_ERROR,     //< Print warning, and return error code
  CBT_EXIT_ON_ERROR      //< Print warning and exit
} ;

  
//#define CBT_ERROR(errno, reason, ...) return cbt_error(errno,__FILE__, __LINE__, __func__, reason, __VA_ARGS__)
#define CBT_ERROR(errno, reason ) cbt_error(errno,__FILE__, __LINE__, __func__, reason )

int
cbt_error(const int errno, 
          const char file[], 
          const int line, 
          const char func[], 
          const char reason[] );




// return the index of the minimal value of vec
inline 
size_t cbt_argmin(const size_t len, const int vec[len]);

// return the index of the minimal value of vec 
inline size_t cbt_argmin_dbl(const size_t len, const double vec[len]);

// return the index of the maximal value of vec 
inline size_t cbt_argmax(const size_t len, const int vec[len]);

// return the index of the maximal value of vec 
inline size_t cbt_argmax_dbl(const size_t len, const double vec[len]);

// Divide each element of vec by the total of vec.
// Return the prenormalized total of vec.
double cbt_normalize(const size_t len, double vec[len]);

// Generate a random number from the system clock.  
unsigned long int  cbt_ran_clock(void);

// Print an error message and exit with failure. 
void cbt_die(const char *comment, ...);

// Divide each element of vec by the total of vec.
// Return the prenormalized total of vec.
double cbt_normalize(const size_t len, double vec[len]);

// Aritmetric mean
double cbt_mean(const size_t len, const double vec[len]);


double 
cbt_variance(const size_t len,
             const double data[]);

double 
cbt_stddev(const size_t len,
           const double data[]); 

double 
cbt_entropy( const size_t len,  
             const double * prob);

// Allocate a 3 dimensional array of integers on the heap. 
int *** 
cbt_mat3d_alloc( int x, int y, int z );

// Free a 3d array allocated by cbt_mat3d_alloc.
int ***
cbt_mat3d_free( int *** mat2free );





#endif /* __GSL_UTIL_H__ */
